--- src/iface.c.orig	2010-02-22 02:26:46.000000000 +0600
+++ src/iface.c	2010-11-21 20:12:44.000000000 +0600
@@ -19,7 +19,10 @@
 #include "netgraph.h"
 #include "util.h"
 
+#include <sys/types.h>
 #include <sys/sockio.h>
+#include <sys/sysctl.h>
+#include <sys/param.h>
 #include <net/if.h>
 #include <net/if_types.h>
 #include <net/if_dl.h>
@@ -58,6 +61,10 @@
 #include <pcap.h>
 #endif
 
+#include <stdio.h>
+#include <stdlib.h>
+#include <string.h>
+
 /*
  * DEFINITIONS
  */
@@ -73,7 +80,8 @@
     SET_UP_SCRIPT,
     SET_DOWN_SCRIPT,
     SET_ENABLE,
-    SET_DISABLE
+    SET_DISABLE,
+    SET_DESC
   };
 
 /*
@@ -156,6 +164,8 @@
 	IfaceSetCommand, NULL, 2, (void *) SET_ENABLE },
     { "disable [opt ...]",		"Disable option",
 	IfaceSetCommand, NULL, 2, (void *) SET_DISABLE },
+    { "description {string}",		"Interface description",
+	IfaceSetCommand, NULL, 2, (void *) SET_DESC },
     { NULL },
   };
 
@@ -224,6 +234,15 @@
 			    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }}}
 static const struct in6_addr in6mask128 = IN6MASK128;
 
+void
+IfaceFreeDescription(IfaceState iface)
+{
+  if (iface->description)
+	free(iface->description);
+  if (iface->desc_template)
+	free(iface->desc_template);
+  iface->description = iface->desc_template = NULL;
+}
 
 /*
  * IfaceInit()
@@ -247,6 +266,7 @@
   SLIST_INIT(&iface->ss[0]);
   SLIST_INIT(&iface->ss[1]);
 #endif
+  IfaceFreeDescription(iface);
 }
 
 /*
@@ -259,6 +279,10 @@
     IfaceState	const iface = &b->iface;
 
     memcpy(iface, &bt->iface, sizeof(*iface));
+    if (bt->iface.desc_template)
+	iface->desc_template = strdup(bt->iface.desc_template);
+    if (bt->iface.description)
+	iface->description = strdup(bt->iface.description);
 }
 
 /*
@@ -494,6 +518,9 @@
   /* Bring up system interface */
   IfaceChangeFlags(b, 0, IFF_UP | (ready?0:IFF_LINK0));
 
+  /* Set interface description */
+  IfaceSetDescription(b);
+
   /* Send any cached packets */
   IfaceCacheSend(b);
 
@@ -517,6 +544,9 @@
 
   Log(LG_IFACE, ("[%s] IFACE: Down event", b->name));
 
+  /* Reset interface description */
+  IfaceSetDescription(b);
+
   /* Bring down system interface */
   IfaceChangeFlags(b, IFF_UP | IFF_LINK0, 0);
 
@@ -597,6 +627,8 @@
       PATH_IPFW, cb);
 #endif /* USE_IPFW */
 
+  u_rangeclear(&iface->self_addr);
+  u_addrclear(&iface->peer_addr);
 }
 
 /*
@@ -1367,6 +1399,19 @@
       DisableCommand(ac, av, &iface->options, gConfList);
       break;
 
+    case SET_DESC:
+      switch (ac) {
+	case 1:
+	  if (iface->desc_template)
+	    free(iface->desc_template);
+	  iface->desc_template = *av[0] ? strdup(av[0]) : NULL;
+	  IfaceSetDescription(ctx->bund);
+	  break;
+	default:
+	  return(-1);
+      }
+      break;
+
     default:
       assert(0);
   }
@@ -1393,6 +1438,7 @@
 
     Printf("Interface configuration:\r\n");
     Printf("\tName            : %s\r\n", iface->ifname);
+    Printf("\tDescription     : %s\r\n", iface->description ? iface->description : "<none>");
     Printf("\tMaximum MTU     : %d bytes\r\n", iface->max_mtu);
     Printf("\tIdle timeout    : %d seconds\r\n", iface->idle_timeout);
     Printf("\tSession timeout : %d seconds\r\n", iface->session_timeout);
@@ -1500,6 +1546,232 @@
     return(0);
 }
 
+/* Determine whether SIOCSIFDESCR symbol is accessible */
+#if (__FreeBSD_version >= 800503 && __FreeBSD_version < 900000) \
+    || (__FreeBSD_version >= 900009)
+#define KernelIfaceDescriptionSupported
+#endif
+
+/*
+ * Set/clear our own interface description accessible in console/web interface
+ * and kernel level interface description if it is available.
+ *
+ * Template may contain conversion specifications:
+ *
+ * %% expands to single % sign;
+ * %a for interface local address;
+ * %A for peer address;
+ * %i for system interface index, useable for SNMP MIB-II lookups;
+ * %I for interface name;
+ * %l for name of bundle's first link
+ * %S for interface status (DoD/UP/DOWN)
+ * %t for type of bundle's first link (pppoe, pptp, l2tp etc.)
+ * %u for self auth name (or dash if self auth name not used)
+ * %U for peer auth name (or dash if peer has not authenticated)
+ */
+void IfaceSetDescription(Bund b)
+{
+    IfaceState	const iface = &b->iface;
+    char *dst = NULL;
+    char *limit, *src;
+    int proceed;
+    unsigned desc_size = 0;		/* make compiler happy */
+    unsigned ifdescr_maxlen = 1024;	/* used as limit for old kernels */
+
+#ifdef KernelIfaceDescriptionSupported
+    static int mib[2] = { -1, 0 };	/* MIB for net.ifdescr_maxlen */
+    struct ifreq ifr;
+    size_t ifdescr_maxlen_len = sizeof(ifdescr_maxlen);
+    size_t miblen = sizeof(mib) / sizeof(mib[0]);
+    int s;
+
+    /*
+     * Check whether running kernel supports interface description.
+     * Perform the check only once.
+     */
+    if (*mib<0 && sysctlnametomib ("net.ifdescr_maxlen", &mib[0], &miblen) < 0) {
+      mib[0] = 0;
+      Perror("[%s] IFACE: kernel does not support interface description",
+    	     b->name);
+    }
+
+    /*
+     * Fetch net.ifdescr_maxlen value every time to catch up with changes
+     */
+    if (*mib &&
+	sysctl (mib, 2, &ifdescr_maxlen, &ifdescr_maxlen_len, NULL, 0) < 0) {
+      /* unexpected error from the kernel, use default value */
+      Perror("[%s] IFACE: sysctl (\"net.ifdescr_maxlen\")", b->name);
+      ifdescr_maxlen = 1024;
+    }
+#endif
+
+    if ((src = iface->desc_template)) {
+      dst = iface->description;
+      if (dst) {
+        iface->description = NULL;
+	free(dst);
+      }
+      if ((iface->description = dst = malloc (ifdescr_maxlen)) == NULL) {
+        Log(LG_IFACE2, ("[%s] IFACE: no memory for interface %s description",
+          b->name, iface->ifname));
+        return;
+      }
+
+      /* ifdescr_maxlen includes terminating zero */
+      limit = dst + ifdescr_maxlen - 1; 
+
+      /*
+       * Perform template expansion
+       */
+      proceed = 1;
+      while (proceed && *src && dst < limit)
+      {
+	if (*src != '%') {	/* ordinary symbol, just copy it and proceed */
+	  *dst++ = *src++;
+	  continue;
+	}
+	if (!*(src+1)) {	/* '%' at the end of template, just copy */
+	  *dst++ = *src++;
+	  continue;
+	}
+	switch(*++src) {	/* expand */
+	  case '%':		/* %% got replaced with single % */
+	    *dst++ = *src;
+	    break;
+
+#define DST_COPY(a)				\
+  do { const char *temp = a;			\
+    if (temp && *temp) {			\
+      if ((dst + strlen (temp)) <= limit) {	\
+	dst = stpcpy (dst, temp);		\
+      } else {					\
+	proceed = 0;				\
+      }						\
+    } else {					\
+      *dst++ = '-';				\
+    }						\
+  } while(0)
+
+	  /* self address */
+	  case 'a':
+	    {
+	      char *sep, buf[19]; /* enough for XXX.XXX.XXX.XXX/XX */
+	      u_rangetoa (&iface->self_addr, buf, sizeof(buf));
+	      /* cut netmask */
+	      if ((sep = strchr(buf, '/')))
+	        *sep = '\0';
+	      DST_COPY(buf);
+	    }
+	    break;
+	  /* peer address */
+	  case 'A':
+	    {
+	      char buf[19];	/* enough for XXX.XXX.XXX.XXX/XX */
+	      u_addrtoa (&iface->peer_addr, buf, sizeof(buf));
+	      DST_COPY(buf);
+	    }
+	    break;
+	  /* interface index */
+	  case 'i':
+	    {
+	      char buf[21];	/* enough for 64bit unsigned int */
+	      snprintf (buf, sizeof(buf), "%u", iface->ifindex);
+	      DST_COPY(buf);
+	    }
+	    break;
+	  /* interface name */
+	  case 'I':
+	    DST_COPY(iface->ifname);
+	    break;
+	  /* first link name */
+	  case 'l':
+	    DST_COPY(b->links[0] ? b->links[0]->name : NULL);
+	    break;
+	  /* interface status */
+	  case 'S':
+	    DST_COPY(iface->up ? (iface->dod ? "DoD" : "UP") : "DOWN");
+	    break;
+	  /* first link type */
+	  case 't':
+	    DST_COPY(b->links[0] ? b->links[0]->type->name : NULL);
+	    break;
+	  /* self auth name */
+	  case 'u':
+	    DST_COPY(b->links[0] ? b->links[0]->lcp.auth.conf.authname : NULL);
+	    break;
+	  /* peer auth name */
+	  case 'U':
+	    DST_COPY(b->params.authname);
+	    break;
+#undef DST_COPY
+	  default: /* unrecognized specification, just copy */
+	    *dst++ = '%';
+	    if (dst < limit)
+		*dst++ = *src;
+	} /* switch(*++src) */
+	++src;
+      } /* while */
+      *dst = '\0';
+
+      /* includes terminating zero */
+      desc_size = dst - iface->description + 1;
+      /* free unneeded part of memory */
+      if (desc_size != ifdescr_maxlen)
+        iface->description = dst = reallocf(iface->description, desc_size); 
+      if (!dst) {
+        Log(LG_IFACE2, ("[%s] IFACE: no memory for interface %s description",
+			b->name, iface->ifname));
+        return;
+      }
+    } else { /* iface->desc_template == NULL */
+      if (iface->description) {
+        free (iface->description);
+        iface->description = NULL;
+      }
+    }
+
+#ifdef KernelIfaceDescriptionSupported
+    if (*mib == 0)
+	return;		/* do not bother kernel if it is too old */
+
+    if (!*iface->ifname)
+	return;		/* we have not set system interface name yet */
+
+    strlcpy (ifr.ifr_name, iface->ifname, IFNAMSIZ);
+
+    if (!dst || !*dst) {   	   /* empty description or clearing request */
+        ifr.ifr_buffer.length = 0; /* let's remove it from kernel interface */
+        ifr.ifr_buffer.buffer = NULL;
+    }
+    else {
+      ifr.ifr_buffer.length = desc_size;
+      ifr.ifr_buffer.buffer = dst;
+    }
+
+    if ((s = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
+	Perror("[%s] IFACE: Can't get socket to set description for %s",
+		b->name, iface->ifname);
+	return;
+    }
+
+    if (dst) {
+      Log(LG_IFACE2, ("[%s] IFACE: Set %s description: %s",
+			b->name, iface->ifname, dst));
+    } else {
+      Log(LG_IFACE2, ("[%s] IFACE: Clear %s description",
+			b->name, iface->ifname));
+    }
+
+    if (ioctl(s, SIOCSIFDESCR, (caddr_t)&ifr) < 0) {
+	Perror("[%s] IFACE: ioctl(%s, %s, \"%s\")",
+		b->name, iface->ifname, "SIOCSIFDESCR",
+		dst ? dst : "" );
+    }
+    close (s);
+#endif
+}
+
 /*
  * IfaceSetMTU()
  *
